// docParse
//
// @(#)main.go  星期四, 八月 15, 2024
// Copyright(c) 2024, huiruli@Tencent. All rights reserved.

package main

import (
	"awesomeProject1/model"
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"net/url"
	"os"
	"path/filepath"
	"strings"

	"github.com/google/uuid"
	"github.com/spf13/cast"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	tchttp "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/http"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
	"github.com/tencentyun/cos-go-sdk-v5"
	"github.com/tmaxmax/go-sse"
)

const (
	EndPoint        = "lke.tencentcloudapi.com"
	SecretID        = ""
	SecretKey       = ""
	BotBizID        = ""         // BotBizID通常是19位的一个数字,如果不知道如何获取请参考：https://cloud.tencent.com/document/product/1759/109469 第三项
	BotAppKey       = ""         // BotAppKey是一个8位的字符串,如果不知道如何获取请参考：https://cloud.tencent.com/document/product/1759/109469 第三项
	TypeKeyRealtime = "realtime" // 实时文档
	TypeKeyOffline  = "offline"  // 离线文档

	DocParseUrl = "https://wss.lke.cloud.tencent.com/v1/qbot/chat/docParse"
	sseUrl      = "https://wss.lke.cloud.tencent.com/v1/qbot/chat/sse"
	Region      = "ap-guangzhou"
)

func main() {
	filePath := "./致橡树.txt"
	//filePath := "./小楷.jpeg"

	fileName := filepath.Base(filePath)
	fileExt := strings.Trim(filepath.Ext(filePath), ".")
	finfo, _ := os.Stat(filePath)
	fileSize := finfo.Size()
	fmt.Printf("文件名：%+v,文件类型：%+v,文件大小：%+v\n", fileName, fileExt, fileSize)

	// 1. 临时密钥的获取，请注意，参数组合不同得到的临时密钥权限不同，会影响后面上传cos和文件解析的结果。常见问题如： 上传cos报错403， 实时文档解析报错 Invalid-URL
	// 可参考 https://cloud.tencent.com/document/product/1759/116238 的参数组合，或者在遇到需要上传文件的地方F12抓包DescribeStorageCredential接口，看下参数组合
	isPublic := false
	// 请注意，该场景为对话接口上传，isPublic做了特殊判断，其他场景请参考上上面文档确定图片是否需要特殊处理
	if fileExt == "jpg" || fileExt == "jpeg" || fileExt == "png" || fileExt == "bmp" {
		isPublic = true
	}
	storageReq := fmt.Sprintf(`{ "BotBizId":"%s" ,"FileType":"%s","IsPublic": %v,"TypeKey":"%s"}`, BotBizID, fileExt, isPublic, TypeKeyRealtime)
	fmt.Printf("getDescribeStorageCredential req:%+v", storageReq)
	resp, err := commonReq("DescribeStorageCredential", storageReq, "POST")
	if err != nil {
		return
	}

	var storageResp model.DescribeStorageCredential
	json.Unmarshal([]byte(resp), &storageResp)

	tmpSecretKey := storageResp.Response.Credentials.TmpSecretKey
	tmpSecretID := storageResp.Response.Credentials.TmpSecretId
	tmpToken := storageResp.Response.Credentials.Token
	bucket := storageResp.Response.Bucket
	region := storageResp.Response.Region
	uploadPath := storageResp.Response.UploadPath // 该地址对应cos中存储位置

	// 拼接一个cos地址，用于文件问答时的传参
	bucketURL := "https://" + bucket + "." + storageResp.Response.Type + "." + region + ".myqcloud.com" + uploadPath

	fmt.Printf(" \n ======= 1.getDescribeStorageCredential success =====\n")
	fmt.Printf("\ntemSecretID:%s,tmpSecretKey:%s,token:%s,uploadPath:%s,bucketURL:%s\n\n", tmpSecretID, tmpSecretKey, tmpToken, uploadPath, bucketURL)

	// 2. 将文件上传到cos，请注意，使用的是临时密钥建立的链接， 传入的是三个参数，其中tmpToken是必填的
	u, _ := url.Parse(bucketURL)
	b := &cos.BaseURL{BucketURL: u}
	client := cos.NewClient(b, &http.Client{
		Transport: &cos.AuthorizationTransport{
			SecretID:     tmpSecretID,
			SecretKey:    tmpSecretKey,
			SessionToken: tmpToken, // 尤其注意这个参数是必填的
		},
	})

	opt := &cos.ObjectPutOptions{
		ObjectPutHeaderOptions: &cos.ObjectPutHeaderOptions{
			// ContentType: "text/html",  // 此处要么不要填写，要么按照实际文件类型填写
		},
	}

	response, err := client.Object.PutFromFile(context.Background(), uploadPath, filePath, opt)
	if err != nil {
		fmt.Printf("\nERROR: cosUploadFle Failed! err:%+v", err)
		return
	}

	eTag := response.Header.Get("Etag")
	cosHash := response.Header.Get("X-Cos-Hash-Crc64ecma")

	fmt.Printf(" \n ======= 2.cosUploadFile success =====\n")
	fmt.Printf("eTag:%s,cosHash:%s,fileSize:%d\n", eTag, cosHash, fileSize)

	// sessionID很重要，请遵循规则生成，docParse传入的sessionID需和对话时穿入的sessionID保持一致
	sessionID := getSessionID()

	// 请注意，当前系统对图片和文件支持的类型和大小都有限制，随着产品能力的迭代，可能会放开文件类型和文件大小的限制
	// 请参考知识引擎页面对话窗口对图片，文件类型大小的支持
	if fileExt == "jpg" || fileExt == "jpeg" || fileExt == "png" || fileExt == "bmp" {
		// 包含图片的content请以markdown格式包裹
		// 示例："content": "![](https://lke-realtime-1251316161.cos.ap-guangzhou.myqcloud.com/public/1746827241600319488/1780784842443587584/image/xxxx.jpg)"
		content := "请描述一下图片信息"
		content = fmt.Sprintf("%s![](%s)", content, bucketURL)
		sseRequest := model.SseSendEvent{
			Content:       content,
			BotAppKey:     BotAppKey,
			VisitorBizID:  uuid.NewString(),
			SessionID:     sessionID,
			VisitorLabels: nil,
		}
		// 请注意，这个sseChat主要是快速验证效果，该函数未处理任何事件， 请接入方参考https://cloud.tencent.com/document/product/1759/105560，结合自己的业务处理相关逻辑
		// 尤其注意，需要判断reply事件中的is_evil字段， is_evil=true,表示被风控拦截了，当前对用户的输入及大模型的输出都接入了风控策略
		sseChat(sseRequest)
	} else {
		// 请注意文档解析用的sessionID需要和对话的sessionID保持一致
		// docParse
		docParseReq := model.DocParseRequest{
			SessionId: sessionID,
			RequestId: sessionID,
			CosBucket: bucket,
			FileType:  fileExt,
			FileName:  fileName,
			CosUrl:    uploadPath,
			ETag:      eTag,
			CosHash:   cosHash,
			Size:      fmt.Sprintf("%d", fileSize),
			BotAppKey: BotAppKey,
		}
		docID, err := docParse(DocParseUrl, docParseReq)
		if err != nil {
			fmt.Printf("\nERROR: DocParse Failed! req:%+v,err:%+v", docParseReq, err)
			return
		}
		fmt.Printf("\ndocID:%+v", docID)
		fmt.Printf(" \n ======= DocParse success =====\n")

		// 调用sse接口对话
		sseRequest := model.SseSendEvent{
			Content:       "",
			BotAppKey:     BotAppKey,
			VisitorBizID:  sessionID,
			SessionID:     sessionID,
			VisitorLabels: nil,
			FileInfos: []model.FileInfo{{
				DocId:    docID,
				FileName: fileName,
				FileType: fileExt,
				FileSize: cast.ToString(fileSize),
				FileUrl:  uploadPath,
			}},
		}
		// 请注意，这个sseChat主要是快速验证效果，该函数未处理任何事件， 请接入方参考https://cloud.tencent.com/document/product/1759/105560，结合自己的业务处理相关逻辑
		// 尤其注意，需要判断reply事件中的is_evil字段， is_evil=true,表示被风控拦截了，当前对用户的输入及大模型的输出都接入了风控策略
		sseChat(sseRequest)
	}

	fmt.Printf(" \n\n========== done ===========\n\n")
}

func uploadToCos() {

}

func commonReq(action, params, reqMethod string) (string, error) {
	credential := common.NewCredential(SecretID, SecretKey)
	cpf := profile.NewClientProfile()
	cpf.HttpProfile.Endpoint = EndPoint
	cpf.HttpProfile.ReqMethod = reqMethod
	client := common.NewCommonClient(credential, Region, cpf).WithLogger(log.Default())

	request := tchttp.NewCommonRequest("lke", "2023-11-30", action)

	err := request.SetActionParameters(params)
	if err != nil {
		fmt.Println(err)
		return "", err
	}

	response := tchttp.NewCommonResponse()
	err = client.Send(request, response)
	if err != nil {
		fmt.Printf("ERROR: Failed Invoke API:action:%s,params:%s,reqMethod:%s,err:%+v", action, params, reqMethod, err)
		return "", err
	}

	return string(response.GetBody()), nil

}

func sseChat(request model.SseSendEvent) {
	fmt.Printf("\n\n================ 开始对话 ==============\n\n")
	requestJSON, err := json.Marshal(request)
	if err != nil {
		fmt.Printf("Error marshaling request body: %v\n", err)
		return
	}

	req, err := http.NewRequest("POST", sseUrl, bytes.NewBuffer(requestJSON))
	if err != nil {
		fmt.Printf("Error creating request: %v\n", err)
		return
	}
	req.Header.Set("Content-Type", "application/json")

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		fmt.Printf("Error sending request: %v\n", err)
		return
	}
	defer resp.Body.Close()

	for ev, err := range sse.Read(resp.Body, nil) {
		if err != nil {
			fmt.Println(err)
			return
		}
		// 接入方请按照https://cloud.tencent.com/document/product/1759/105560自行处理各种返回事件
		fmt.Println(ev)
		fmt.Println()
	}
}

// 文档解析
func docParse(url string, requestBody model.DocParseRequest) (string, error) {
	requestJSON, err := json.Marshal(requestBody)
	if err != nil {
		return "", fmt.Errorf("error marshaling request body: %v", err)
	}

	req, err := http.NewRequest("POST", url, bytes.NewBuffer(requestJSON))
	if err != nil {
		return "", fmt.Errorf("error creating request: %v", err)
	}
	req.Header.Set("Content-Type", "application/json")

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return "", fmt.Errorf("error sending request: %v", err)
	}
	defer resp.Body.Close()

	for ev, err := range sse.Read(resp.Body, nil) {
		if err != nil {
			fmt.Println(err)
			return "", err
		}
		fmt.Printf("\n==========\n")
		var parseEvent model.DocParseEvent
		json.Unmarshal([]byte(ev.Data), &parseEvent)
		if parseEvent.Payload.IsFinal {
			fmt.Printf("Parsing completed for doc_id: %s with status: %s\n", parseEvent.Payload.DocID, parseEvent.Payload.Status)
			return parseEvent.Payload.DocID, nil
		} else {
			fmt.Printf("Parsing in progress for doc_id: %s, process: %d%%\n", parseEvent.Payload.DocID, parseEvent.Payload.Process)
		}
		if parseEvent.Payload.Status == "FAILED" {
			fmt.Printf("docParse Failed! err:%+v\n", parseEvent.Payload.ErrorMessage)
			return "", fmt.Errorf("docParse Failed! err:%+v", parseEvent.Payload.ErrorMessage)
		}
	}

	return "", nil
}

func getSessionID() string {
	return uuid.NewString()
}
