import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.tencentcloudapi.common.Credential;
import com.tencentcloudapi.common.exception.TencentCloudSDKException;
import com.tencentcloudapi.common.profile.ClientProfile;
import com.tencentcloudapi.common.profile.HttpProfile;
import com.tencentcloudapi.common.CommonClient;

import com.qcloud.cos.COSClient;
import com.qcloud.cos.ClientConfig;
import com.qcloud.cos.auth.BasicSessionCredentials;
import com.qcloud.cos.model.PutObjectRequest;
import com.qcloud.cos.model.PutObjectResult;
import com.qcloud.cos.model.ObjectMetadata;
import com.qcloud.cos.region.Region;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import java.util.logging.StreamHandler;


import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

public class Sample {

    private static final Logger LOGGER = Logger.getLogger(Sample.class.getName());

    // 配置日志格式
    static {
        StreamHandler handler = new StreamHandler(System.out, new SimpleFormatter());
        LOGGER.addHandler(handler);
        LOGGER.setLevel(Level.INFO);
    }

    private static final String EndPoint = "lke.tencentcloudapi.com";
    private static final String SecretID = "";
    private static final String SecretKey = "";
    private static final String BotBizID = ""; // BotBizID通常是19位的一个数字,如果不知道如何获取请参考：https://cloud.tencent.com/document/product/1759/109469 第三项
    private static final String BotAppKey = ""; // BotAppKey是一个8位的字符串,如果不知道如何获取请参考：https://cloud.tencent.com/document/product/1759/109469 第三项
    private static final String REGION = "ap-guangzhou";
    private static final String TypeKey = "realtime";
    private static final String DocParseURL = "https://wss.lke.cloud.tencent.com/v1/qbot/chat/docParse";
    private static final String ChatSSEURL = "https://wss.lke.cloud.tencent.com/v1/qbot/chat/sse";

    public static void main(String[] args) {
        try {
//              String filePath = "/Users/reinhold/Desktop/对话 sse/chat_with_file_or_img_java/src/main/java/com/tencent/小楷.jpeg";
            String filePath = "/Users/reinhold/Desktop/对话 sse/chat_with_file_or_img_java/src/main/java/com/tencent/致橡树.txt";
            String fileExt = getFileExtension(filePath);

            // 获取文件名,文件大小
            Path fileName  = Paths.get(filePath).getFileName();
            File localFile = new File(filePath);
            Long fileSize = localFile.length();



            // 生成会话 ID
            // session_id很重要，请遵循规则生成，docParse传入的session_id需和对话时传入的session_id保持一致
            // visitorBizID 客户结合自己的业务按规则生成
            String sessionID = UUID.randomUUID().toString();
            String visitorBizID = UUID.randomUUID().toString();

            LOGGER.info(() -> "File Information: " + filePath + ", " + fileExt + ", fileName: " + fileName);

            // 请注意，对话是上传图片和知识库上传图片限制的图片格式和大小随着产品能力的迭代会增加或者扩大，请参考知识引擎页面调整图片的支持范围
            boolean isPublic = isPublicFile(fileExt);

            // 临时密钥的获取，请注意，参数组合不同得到的临时密钥权限不同，会影响后面上传cos和文件解析的结果。常见问题如： 上传cos报错403， 实时文档解析报错 Invalid-URL
            // 可参考 https://cloud.tencent.com/document/product/1759/116238 的参数组合，或者在遇到需要上传文件的地方F12抓包DescribeStorageCredential接口，看下参数组合
            // 请注意，该场景为对话接口上传，is_public做了特殊判断，其他场景请参考上上面文档确定图片是否需要特殊处理
            Map<String, String> credentials = getTemporaryCredentials(fileExt,isPublic);

            // 上传文件到cos
            // 请注意，需要使用临时密钥建立cos_clint, 不同的语言方式不一样，使用其他语言可自行参考cos的sdk
            // 目前已知至少包含三个参数【secret_key,secret_id,token】, 该参数请从获取临时密钥接口返回的数据中获取
            Map<String,String> cosResult = uploadFileToCOS(filePath, credentials);

            // 图片处理逻辑, 图片的类型和大小限制的扩充，请关注知识引擎发版公告或者页面上的限制
            if (isImageFile(fileExt)) {
                String content = "请帮我描述一下这张图片";
                String markdownContent = String.format("%s![](%s)", content, cosResult.get("CosFinalURL").toString());

                 Map<String, Object> sseRequestData = new HashMap<>();
                 sseRequestData.put("content",markdownContent);
                 sseRequestData.put("bot_app_key", BotAppKey);
                 sseRequestData.put("visitor_biz_id", visitorBizID);
                 sseRequestData.put("session_id", sessionID);
                sseRequestData.put("visitor_labels", new ArrayList<>());
                sseRequestData.put("request_id", sessionID);
                // 开始对话
                // 请注意，这个sseChat主要是快速验证效果，请接入方参考https://cloud.tencent.com/document/product/1759/105560，结合自己的业务选择http sse或者websocket处理相关逻辑
                // 尤其注意，需要判断reply事件中的is_evil字段， is_evil=true,表示被风控拦截了，当前对用户的输入及大模型的输出都接入了风控策略
                sendSseRequest(sseRequestData);
            } else {
                String docId = parseDocument(credentials, filePath,String.valueOf(fileSize), fileExt,cosResult);
                System.out.print(docId);


                // 创建 file_infos 列表
                List<Map<String, String>> fileInfos = new ArrayList<>();
                Map<String, String> fileInfo = new HashMap<>();
                fileInfo.put("doc_id", docId);
                fileInfo.put("file_name", String.valueOf(fileName));
                fileInfo.put("file_type", fileExt);
                fileInfo.put("file_size", String.valueOf(fileSize));
                fileInfo.put("file_url", cosResult.get("CosFinalURL"));
                fileInfos.add(fileInfo);

                // 创建 sse_request_data
                Map<String, Object> sseRequestData = new HashMap<>();
                sseRequestData.put("content", "");
                sseRequestData.put("bot_app_key", BotAppKey);
                sseRequestData.put("visitor_biz_id", sessionID);
                sseRequestData.put("session_id", sessionID);
                sseRequestData.put("visitor_labels", new ArrayList<>());
                sseRequestData.put("request_id", sessionID);
                sseRequestData.put("file_infos", fileInfos);
                // 开始对话
//   请注意，这个sseChat主要是快速验证效果，请接入方参考https://cloud.tencent.com/document/product/1759/105560，结合自己的业务选择http sse或者websocket处理相关逻辑
//	 尤其注意，需要判断reply事件中的is_evil字段， is_evil=true,表示被风控拦截了，当前对用户的输入及大模型的输出都接入了风控策略
                sendSseRequest(sseRequestData);
            }
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, "An error occurred", e);
        }
    }

    private static String getFileExtension(String filePath) {
        Path path = Paths.get(filePath);
        Path fileName = path.getFileName();
        if (fileName == null) {
            return "";
        }
        String fullName = fileName.toString();
        int dotIndex = fullName.lastIndexOf('.');
        return (dotIndex != -1 && dotIndex < fullName.length() - 1) ? fullName.substring(dotIndex + 1) : "";
    }

    private static boolean isPublicFile(String fileExt) {
        return List.of("jpg", "jpeg", "png", "bmp").contains(fileExt.toLowerCase());
    }

    private static boolean isImageFile(String fileExt) {
        return isPublicFile(fileExt);
    }

    private static Map<String, String> getTemporaryCredentials(String fileExt,Boolean isPublic) throws TencentCloudSDKException {
        Credential cred = new Credential(SecretID, SecretKey);
        HttpProfile httpProfile = new HttpProfile();
        httpProfile.setEndpoint(EndPoint);
        ClientProfile clientProfile = new ClientProfile();
        clientProfile.setHttpProfile(httpProfile);
        CommonClient client = new CommonClient("lke", "2023-11-30", cred, REGION, clientProfile);

        String params = String.format("{\"BotBizId\":\"%s\",\"FileType\":\"%s\",\"TypeKey\":\"%s\",\"IsPublic\":%b}",
                BotBizID, fileExt, TypeKey, isPublic);
        String resp = client.call("DescribeStorageCredential", params);

        JsonParser parser = new JsonParser();
        JsonObject jsonObject = parser.parse(resp).getAsJsonObject();
        JsonObject credentials = jsonObject.get("Response").getAsJsonObject().get("Credentials").getAsJsonObject();

        return Map.of(
                "TmpSecretId", credentials.get("TmpSecretId").getAsString(),
                "TmpSecretKey", credentials.get("TmpSecretKey").getAsString(),
                "Token", credentials.get("Token").getAsString(),
                "Bucket", jsonObject.get("Response").getAsJsonObject().get("Bucket").getAsString(),
                "UploadPath", jsonObject.get("Response").getAsJsonObject().get("UploadPath").getAsString(),
                "Region", jsonObject.get("Response").getAsJsonObject().get("Region").getAsString(),
                "Type", jsonObject.get("Response").getAsJsonObject().get("Type").getAsString()
        );
    }

    private static  Map<String, String> uploadFileToCOS(String filePath, Map<String, String> credentials) throws IOException {
        ObjectMetadata metadata = new ObjectMetadata();
        // 请注意，需要使用临时密钥建立cos_clint, 不同的语言方式不一样，使用其他语言可自行参考cos的sdk
        // 目前已知至少包含三个参数【secret_key,secret_id,token】, 该参数请从获取临时密钥接口返回的数据中获取
        BasicSessionCredentials cosCred = new BasicSessionCredentials(
                credentials.get("TmpSecretId"),
                credentials.get("TmpSecretKey"),
                credentials.get("Token")
        );
        Region region = new Region(credentials.get("Region"));
        ClientConfig clientConfig = new ClientConfig(region);
        COSClient cosClient = new COSClient(cosCred, clientConfig);

        File localFile = new File(filePath);
        PutObjectRequest putObjectRequest = new PutObjectRequest(
                credentials.get("Bucket"),
                credentials.get("UploadPath"),
                localFile
        );
        PutObjectResult putObjectResult = cosClient.putObject(putObjectRequest);

        String cosFinalUrl = String.format("https://%s.%s.%s.myqcloud.com%s",
                credentials.get("Bucket"),
                credentials.get("Type"),
                credentials.get("Region"),
                credentials.get("UploadPath")
        );

        String eTag = putObjectResult.getETag();
        String cosHash = putObjectResult.getCrc64Ecma();

        return Map.of(
                "CosFinalURL",cosFinalUrl,
                "Etag", eTag,
                "CosHash", cosHash
        );
    }

    private static String parseDocument(Map<String, String> credentials, String filePath,String fileSize, String fileExt, Map<String,String> cosResult) throws IOException, InterruptedException {
        String sessionID = UUID.randomUUID().toString();
        Path path = Paths.get(filePath);
        String fileName = path.getFileName().toString();

        Map<String, Object> parseDocReq = new HashMap<>();
        parseDocReq.put("session_id",sessionID);
        parseDocReq.put("request_id", sessionID);
        parseDocReq.put("cos_bucket", credentials.get("Bucket"));
        parseDocReq.put("file_type", fileExt);
        parseDocReq.put("file_name", fileName);
        parseDocReq.put("cos_url",  credentials.get("UploadPath"));
        parseDocReq.put("e_tag",  cosResult.get("Etag"));
        parseDocReq.put("cos_hash",   cosResult.get("CosHash"));
        parseDocReq.put("size", fileSize);
        parseDocReq.put("bot_app_key",BotAppKey);

        ObjectMapper objectMapper1 = new ObjectMapper();
        String jsonRequestBody = objectMapper1.writeValueAsString(parseDocReq);
        LOGGER.info(() -> "parseDocument Request: " + jsonRequestBody);


        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(DocParseURL))
                .header("Accept", "text/event-stream")
                .POST(HttpRequest.BodyPublishers.ofString(jsonRequestBody))
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
        String responseBody = response.body();
        ObjectMapper objectMapper = new ObjectMapper();
        String docID = "";

        if (response.statusCode() == 200) {
            String[] lines = responseBody.split("\n");
            for (String line : lines) {
                System.out.print("docParseResult: " + line);
                if (line.startsWith("data:")) {
                    JsonNode rootNode = objectMapper.readTree(line.substring(5).trim());
                    String tmpDocID = rootNode.path("payload").path("doc_id").asText();

                    Boolean isFinal =  rootNode.path("payload").path("is_final").asBoolean();
                    String status = rootNode.path("payload").path("status").asText();
                    String errorMessage = rootNode.path("payload").path("error_message").asText();
                    if (isFinal) {
                        docID = tmpDocID;
                        LOGGER.info(() -> "Extracted doc_id: " + tmpDocID);
                    }
                    if (status == "FAILED") {
                        LOGGER.info(() -> "解析失败,Extracted errorMessage: " + errorMessage);
                    }
                }


            }
        } else {
            LOGGER.severe("Failed to parse document: " + responseBody);
        }
        return docID;
    }


    private static void sendSseRequest(Map<String, Object> sseRequestData) throws JsonProcessingException, IOException, InterruptedException {
        ObjectMapper objectMapper = new ObjectMapper();
        String jsonRequestBody = objectMapper.writeValueAsString(sseRequestData);
        LOGGER.info(() -> "SSE Request: " + jsonRequestBody);


        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(ChatSSEURL))
//                .header("Accept", "text/event-stream")
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers.ofString(jsonRequestBody))
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
        LOGGER.info(() -> "Response Code: " + response.statusCode());
        LOGGER.info(() -> "Response Body: " + response.body());
    }
}